%% DICTIONARY LEARNING FOR IMAGE DENOISING
%
%   Centre for Digital Music, Queen Mary, University of London.
%   This file copyright 2010 Ivan Damnjanovic.
%
%   This program is free software; you can redistribute it and/or
%   modify it under the terms of the GNU General Public License as
%   published by the Free Software Foundation; either version 2 of the
%   License, or (at your option) any later version.  See the file
%   COPYING included with this distribution for more information.
%   
%   This file contains an example of how SMALLbox can be used to test different
%   dictionary learning techniques in Image Denoising problem.
%   This example can be used to test SPAMS for different values of
%   parameter lambda. In no way it represents extensive testing of image
%   denoising. It should only give an idea how SMALL structure can be used
%   for testing.
%
%%

clear all;

%% Load an image
TMPpath=pwd;
FS=filesep;
[pathstr1, name, ext, versn] = fileparts(which('SMALLboxSetup.m'));
cd([pathstr1,FS,'data',FS,'images']);
[filename,pathname] = uigetfile({'*.png;'},'Select a file containin pre-calculated notes');
[pathstr, name, ext, versn] = fileparts(filename);
test_image = imread(filename);
test_image = double(test_image);
cd(TMPpath);
%%

%   number of different values we want to test

n =4;

lambda=zeros(1,n);
time = zeros(2,n);
psnr = zeros(2,n);

for i=1:n
    
    %   Here we want to test time spent and quality of denoising for
    %   different lambda parameters.
    
    lambda(i)=1+i*0.5;
    
    %   Defining Image Denoising Problem as Dictionary Learning Problem.
    
    SMALL.Problem = generateImageDenoiseProblem(test_image);
    SMALL.Problem.name=name;
    %%
    %   Use SPAMS Online Dictionary Learning Algorithm
    %   to Learn overcomplete dictionary (Julien Mairal 2009)
    
    %   Initialising Dictionary structure
    %   Setting Dictionary structure fields (toolbox, name, param, D and time)
    %   to zero values
    
    SMALL.DL(1)=SMALL_init_DL();
    
    %   Defining fields needed for dictionary learning
    
    SMALL.DL(1).toolbox = 'SPAMS';
    SMALL.DL(1).name = 'mexTrainDL';
    
    %   Type 'help mexTrainDL in MATLAB prompt for explanation of parameters.
    
    SMALL.DL(1).param=struct(...
        'D', SMALL.Problem.initdict,...
        'K', SMALL.Problem.p,...
        'lambda', lambda(i),...
        'iter', 200,...
        'mode', 3,...
        'modeD', 0);
    
    %   Learn the dictionary
    
    SMALL.DL(1) = SMALL_learn(SMALL.Problem, SMALL.DL(1));
    
    %   Set SMALL.Problem.A dictionary
    %   (backward compatiblity with SPARCO: solver structure communicate
    %   only with Problem structure, ie no direct communication between DL and
    %   solver structures)
    
    SMALL.Problem.A = SMALL.DL(1).D;
    
    
    %%
    %   Initialising solver structure
    %   Setting solver structure fields (toolbox, name, param, solution,
    %   reconstructed and time) to zero values
    
    SMALL.solver(1)=SMALL_init_solver;
    
    % Defining the parameters needed for sparse representation
    
    SMALL.solver(1).toolbox='ompbox';
    SMALL.solver(1).name='ompdenoise';
    
    %   Denoising the image - SMALL_denoise function is similar to SMALL_solve,
    %   but backward compatible with KSVD definition of denoising
    
    SMALL.solver(1)=SMALL_denoise(SMALL.Problem, SMALL.solver(1));
    
    
    %% show results %%
    %   This will show denoised image and dictionary for all lambdas. If you
    %   are not interested to see it and do not want clutter your screen
    %   comment following line
    
    SMALL_ImgDeNoiseResult(SMALL);
    
    
    time(1,i) = SMALL.DL(1).time;
    psnr(1,i) = SMALL.solver(1).reconstructed.psnr;
    
    clear SMALL
end

%% show time and psnr %%
figure('Name', 'SPAMS LAMBDA TEST');

subplot(1,2,1); plot(lambda, time(1,:), 'ro-');
title('time vs lambda');
subplot(1,2,2); plot(lambda, psnr(1,:), 'b*-');
title('PSNR vs lambda');

